<?php

require_once 'modules/clients/models/UserPackage.php';
require_once 'modules/clients/models/Client_EventLog.php';
require_once 'modules/clients/models/Package_EventLog.php';
require_once 'modules/admin/models/Package.php';
require_once 'modules/admin/models/StatusAliasGateway.php';

/**
 * Packages Controller
 *
 * @category   Action
 * @package    Clients
 * @author     Alberto Vasquez <alberto@clientexec.com>
 * @license    http://www.clientexec.com  ClientExec License Agreement
 * @link       http://www.clientexec.com
 */
class Clients_ProductspublicController extends CE_Controller_Action {

    var $moduleName = "clients";


    public function sendwelcomeemailAction()
    {
        $this->checkPermissions('clients_send_welcome_email');

        $id = $this->getParam('id');
        $userPackage = new UserPackage($id);

        if ( $this->user->getId() == $userPackage->CustomerId ) {
            $userPackageGateway = new UserPackageGateway($this->user);
            $userPackageGateway->sendWelcomeEmailForPackage($userPackage);

            CE_Lib::redirectPage('index.php?fuse=clients&controller=products&view=products', $this->user->lang('Successfully sent welcome email'));
        } else {
            CE_Lib::redirectPermissionDenied($this->user->lang('You can not send the welcome email for this package'));
        }
    }

    public function generaterenewinvoiceAction()
    {
        $this->checkPermissions('billing_renew_package');

        $id = $this->getParam('id');
        $userPackage = new UserPackage($id);

        if ( $userPackage->isPaid() ) {
            $recurringWork = $userPackage->getRecurringFeeEntry();
            if ($recurringWork->GetRecurring()) {
                if ( $this->user->getId() == $userPackage->CustomerId ) {
                    $billingGateway = new BillingGateway( $this->user );
                    $invoicesData = $billingGateway->generateNextInvoice($this->user->getId(), $id);

                    if ($invoicesData !== false && isset($invoicesData["invoiceIDs"]) && count($invoicesData["invoiceIDs"]) > 0) {
                        //if there are many invoices, merge them.
                        if (count($invoicesData["invoiceIDs"]) > 1) {
                            $billingGateway->mergeInvoices( $invoicesData["invoiceIDs"], $invoicesData["invoiceIDs"][0] );
                        }
                        //Redirect to the invoice
                        CE_Lib::redirectPage('index.php?fuse=billing&controller=invoice&view=invoice&id='.$invoicesData["invoiceIDs"][0], $this->user->lang('Please pay this invoice'));
                        return;
                    }
                }
            }
        }
        CE_Lib::redirectPermissionDenied($this->user->lang('You can not renew this package'));
    }

    public function toogleautomaticccchargeAction()
    {
        $this->checkPermissions('billing_automatic_cc_charge');

        $product_id = $this->getParam('id', FILTER_SANITIZE_NUMBER_INT);
        $userPackage = new UserPackage($product_id);

        if ($this->user->getId() != $userPackage->getCustomerId()) {
            $this->error = true;
            $this->message = $this->user->lang('You can not modify automatic credit card charging for this package');
            $this->send();
            return;
        }

        $gateway = new UserPackageGateway($this->user);
        $gateway->toogleAutomaticCCChargeForPackage($userPackage);

        $this->message = $this->user->lang('Successfully modified automatic credit card charging');
        $this->send();
    }

    /**
     * Handles viewing the public product
     * @return [type] [description]
     */
    protected function productAction()
    {
        include_once 'modules/billing/models/Currency.php';

        $this->title = $this->user->lang('My Product');
        $product_id = $this->getParam('id', FILTER_SANITIZE_NUMBER_INT);
        $userPackage = new UserPackage($product_id);

        $this->cssPages = array("templates/default/css/whhg.min.css","templates/default/views/clients/productspublic/product.css");
        $this->jsLibs = array("templates/default/views/clients/productspublic/product.js");

        $package = new Package($userPackage->Plan);

        $this->view->package = array();
        $this->view->package['customFields'] = array();
        $this->view->product_id = $product_id;

        $languages = CE_Lib::getEnabledLanguages();
        include_once 'modules/admin/models/Translations.php';
        $translations = new Translations();
        $languageKey = ucfirst(strtolower($this->user->getLanguage()));

        if (count($languages) > 1) {
            $this->view->package['product']      = $translations->getValue(PRODUCT_NAME, $package->getId(), $languageKey, $package->planname);
            $this->view->package['productGroup'] = $translations->getValue(PRODUCT_GROUP_NAME, $package->productGroup->getId(), $languageKey, $package->productGroup->name);
        } else {
            $this->view->package['product']      = $package->planname;
            $this->view->package['productGroup'] = $package->productGroup->name;
        }

        $this->view->package['productType'] = $userPackage->getProductType();

        $recurringFee = $userPackage->getRecurringFeeEntry();
        $this->view->package['nextBillDate'] = $recurringFee->getNextBillDate();
        if ($this->view->package['nextBillDate'] == "") {
            $this->view->package['nextBillDate'] = $this->user->lang("Not Applicable");
        }

        // get any package custom fields
        $packageCustomFields = $userPackage->customFields["package"];
        while ($row = $packageCustomFields->fetch()) {
            if ( !$row['isadminonly'] ) {
                $data = array();
                $data['name'] = $row['name'];
                $data['value'] = $row['value'];
                if ($row['fieldtype'] == typeDATE) {
                    $data['value'] = ($row['value'] != '0000-00-00')? CE_Lib::db_to_form($row['value'], $this->settings->get('Date Format'), '/') : '';
                } else if ($row['fieldtype'] == typeDROPDOWN) {
                    $options = explode(",", trim($row['dropdownoptions']));
                    foreach ($options as $option) {
                        if (preg_match('/(.*)(?<!\\\)\((.*)(?<!\\\)\)/', $option, $matches)) {
                            if ($data['value'] == $matches[2]) {
                                $data['value'] = $matches[1];
                            }
                        }
                        $selectOptions[] = array($value,$label);
                    }
                }
                $this->view->package['customFields'][] = $data;
            }
        }

        if ( $userPackage->getProductType() == PACKAGE_TYPE_HOSTING ) {
            // Server
            include_once 'modules/admin/models/ServerGateway.php';
            $serverGateway = new ServerGateway();
            $serverID = $userPackage->getCustomField('Server Id');
            $data = array();
            $data['name'] = $this->user->lang('Server');
            $serverName = $serverGateway->getServerNameById($serverID);
            $data['value'] = ($serverName == "") ? $this->user->lang("None selected") : $serverName;
            $this->view->package['customFields'][] = $data;

            // IP Address
            $data = array();
            $data['name'] = $this->user->lang('IP Address');
            $serverip = $userPackage->getCustomField('IP Address');
            $data['value'] = ($serverip == "") ? $this->user->lang("Not defined") : $serverip;
            $this->view->package['customFields'][] = $data;

            $advancedSettings = @unserialize($package->advanced);
            if ( @$advancedSettings['hostingcustomfields'] == 0 ) {

                // Username
                $data = array();
                $data['name'] = $this->user->lang('Username');
                $data['value'] = $userPackage->getCustomField('User Name');
                $this->view->package['customFields'][] = $data;

                if ( $this->user->hasPermission('clients_change_package_password') ) {
                    // Password
                    $data = array();
                    $data['name'] = $this->user->lang('Password');
                    $data['value'] = '<a href="#" id="passwordChange">'.$this->user->lang('Click To Change').'</a>';
                    $data['password'] = true;
                    $this->view->package['customFields'][] = $data;
                }

                // Domain Name
                $data = array();
                $data['name'] = $this->user->lang('Domain Name');
                $data['value'] = $userPackage->getCustomField('Domain Name');
                $this->view->package['customFields'][] = $data;
            }

        } else if ($userPackage->getProductType() == PACKAGE_TYPE_DOMAIN) {
            $this->title = $this->user->lang('Domain Information');
            $product_id = $this->getParam('id', FILTER_SANITIZE_NUMBER_INT);
            $userPackage = new UserPackage($product_id);

            $this->cssPages = array("templates/default/css/whhg.min.css", "templates/default/css/customfields_public.css", "templates/default/views/clients/productspublic/productdomaininfo.css", "templates/default/views/clients/productspublic/product.css");
            $this->jsLibs = array('javascript/customfields.js', "templates/default/views/clients/productspublic/product.js", "templates/default/views/clients/productspublic/productdomaininfo.js");

            $domainNameGateway = new DomainNameGateway($this->user);
            $isRegistered = $domainNameGateway->isDomainRegistered($userPackage);

            $orderId = $userPackage->getCustomField("Registrar Order Id");
            if ($orderId == "") {
                $orderId = $this->user->lang("Unknown");
            }

            $transferId = $userPackage->getCustomField("Transfer Status");
            if ($transferId == "") {
                $transferId = $this->user->lang("Unknown");
            }

            if ( $userPackage->getCustomField("Registration Option") == 1 ) {
                $this->view->isTransfer = true;
                $this->view->eppCode = '';
                $this->view->transferStatus = '';
                $this->view->transferId = '';
            } else {
                $this->view->isTransfer == false;
            }

            $this->view->productId = $product_id;
            $this->view->orderId = '';
            $this->view->registrar = '';
            $this->view->expirationDate = '';
            $this->view->registrationLength = '';
            $this->view->registrationStatus = '';
            $this->view->purchaseStatus = '';
            $this->view->registrarLock = '';

            // not registered yet
            if ( ( $userPackage->getCustomField("Registration Option") != 1 ) || ( $userPackage->getCustomField("Registration Option") == 1 && $userPackage->getCustomField('Transfer Status') == 'Completed' ) ) {
                $this->view->registrationStatus = $this->user->lang("Not Yet Registered");
            }

            // non-completed transfer
            if ( $userPackage->getCustomField("Registration Option") == 1 && $userPackage->getCustomField('Transfer Status') != 'Completed' ) {
                $this->view->transferId = $transferId;
                try  {
                    $transferStatus = $domainNameGateway->getTransferStatus($userPackage);
                }
                catch ( MethodNotImplemented $e ) {
                    $transferStatus = '';
                }
                catch ( Exception $e ) {
                    $transferStatus = $e->getMessage();
                }
                $this->view->transferStatus = $transferStatus;
            }

            // Only show expiration, registration and purchase status if it's a normal reg, or a completed transfer
            if ( ( $userPackage->getCustomField("Registration Option") != 1 ) || ( $userPackage->getCustomField("Registration Option") == 1 && $userPackage->getCustomField('Transfer Status') == 'Completed' ) ) {
                try {
                    $generalInfo = $domainNameGateway->getGeneralInfoViaPlugin($userPackage);
                } catch ( Exception $e ) {
                }
                $this->view->expirationDate = $generalInfo['expires'];
                $this->view->registrationStatus = $generalInfo['registration'];
                $this->view->purchaseStatus = $generalInfo['purchase'];
                $this->view->canToggleRegLock = $this->user->hasPermission('domains_lock');
                try {
                    $this->view->registrarLock = $domainNameGateway->getRegistrarLockViaPlugin($userPackage);

                    $this->view->regLockValue = false;
                    // we need to check the value through the language parser as the function returns the value like that.
                    if ( $this->view->registrarLock == $this->user->lang('Enabled') ) {
                        $this->view->regLockValue = true;
                    }
                } catch ( Exception $e ) {
                }
            }

            if ( $this->settings->get('Hide Registrar Information') != 1 || $this->user->isAdmin() ) {
                $this->view->registrar = $userPackage->getCustomField("Registrar");
                $this->view->orderId = $orderId;
            }

            $period = $domainNameGateway->getPeriod($userPackage);

            $this->view->registrationLength = $period . ' ';
            $this->view->registrationLength .= ($period == 1) ? $this->user->lang("Year") : $this->user->lang("Years");

        }
    }

    protected function savedomainhostrecordsAction()
    {
        $this->checkPermissions('domains_updatedns');

        $product_id = $this->getParam('id', FILTER_SANITIZE_NUMBER_INT);
        $userPackage = new UserPackage($product_id);

        if ( $this->user->getId() != $userPackage->getCustomerId() ) {
            throw new CE_Exception($this->user->lang('You can not update this domain'));
        }

        $zoneArray = array();
        foreach (array_keys($_REQUEST) as $key) {
            if ($key === "hostname_blankrecord") {
                continue;
            }

            if ("hostname_" === substr($key, 0, 9) ) {
                $hostnameidpair = explode("_", $key);
                if ( $hostnameidpair[2] == 'blankrecord') {
                    continue;
                }

                $record = array(
                    'id'       => filter_var($hostnameidpair[2], FILTER_SANITIZE_STRING),
                    'hostname' => filter_var($_REQUEST['hostname_CT_'.$hostnameidpair[2]], FILTER_SANITIZE_STRING),
                    'address'  => filter_var($_REQUEST['hostaddress_CT_'.$hostnameidpair[2]], FILTER_SANITIZE_STRING),
                    'type'     => filter_var($_REQUEST['hosttype_CT_'.$hostnameidpair[2]], FILTER_SANITIZE_STRING)
                );

                // if the field exists, it means it's a current dns entry
                if ( isset($_REQUEST[$hostnameidpair[2]]) ) {
                    $record['new'] = false;
                } else {
                    // this is a new dns entry
                    $record['new'] = true;
                }

                //let's validate
                if ( (trim($record['hostname']) == "") || (trim($record['address']) == "") ) {
                    throw new CE_Exception("Hostname nor Address can be left blank");
                }

                $zoneArray[] = $record;
            }
        }

        $gateway = new DomainNameGateway($this->user);
        $gateway->callPlugin($userPackage, 'setDNS', array('records' => $zoneArray));

        $this->message = $this->user->lang('Successfully updated host records');
        $this->send();

    }

    protected function savedomainnameserversAction()
    {
        $this->checkPermissions('domains_editns');

        $product_id = $this->getParam('id', FILTER_SANITIZE_NUMBER_INT);
        $userPackage = new UserPackage($product_id);

        if ( $this->user->getId() != $userPackage->getCustomerId() ) {
            throw new CE_Exception($this->user->lang('You can not update this domain'));
        }

        $useDefaults = $this->getParam('ns_usedefaults', FILTER_SANITIZE_NUMBER_INT, 0);
        $useDefaults = ($useDefaults == 1) ? true : false;

        $ns = array();
        for ($i = 0; $i < 9; $i++) {
            if (!isset($_POST['ns_'.$i])) {
                continue;
            }
            $ns[$i] = filter_var($_POST['ns_'.$i], FILTER_SANITIZE_STRING);
        }

        $gateway = new DomainNameGateway($this->user);
        $gateway->callPlugin($userPackage, 'setNameServers', array('ns' => $ns, 'default' => $useDefaults));

        $this->message = $this->user->lang('Successfully updated name servers');
        $this->send();
    }

    protected function productdomainnameserversAction()
    {
        $this->title = $this->user->lang('Domain Name Servers');
        $this->checkPermissions('domains_nameservers');

        $product_id = $this->getParam('id', FILTER_SANITIZE_NUMBER_INT);
        $userPackage = new UserPackage($product_id);

        $this->cssPages = array("templates/default/css/whhg.min.css", "templates/default/css/customfields_public.css", "templates/default/views/clients/productspublic/product.css");
        $this->jsLibs = array('javascript/customfields.js', "templates/default/views/clients/productspublic/productdomainnameservers.js", "templates/default/views/clients/productspublic/product.js");

        $this->view->domainNotActive = false;
        $this->view->domainNotRegistered = false;
        $this->view->packageId = $product_id;

        $domainNameGateway = new DomainNameGateway($this->user);
        $isRegistered = $domainNameGateway->isDomainRegistered($userPackage);

        $statusPending = StatusAliasGateway::getInstance($this->user)->getPackageStatusIdsFor(PACKAGE_STATUS_PENDING);
        if ( !$isRegistered ) {
            $this->view->domainNotRegistered = true;
        } else if ($isRegistered && !in_array($userPackage->status, $statusPending)) {
            include_once 'modules/clients/models/ObjectCustomFields.php';

            $datalist = array();

            $data = array();
            $data["id"]           = 'blankrecord';
            $data["name"]         = "";
            $data["ischangeable"] = true;
            $data["isrequired"]   = false;
            $data["ishidden"]     = true;
            $data["fieldtype"]    = (string)typeNAMESERVER;
            $data["value"]        = "";
            ObjectCustomFields::parseFieldToArray($data, $this->user, $this->settings);
            $datalist[] = $data;

            $gateway = new DomainNameGateway($this->user);
            $NSArray = $gateway->getNameServersViaPlugin($userPackage);

            //hide the nameservers if uses default
            $hidefields = (isset($NSArray['usesDefault']) && $NSArray['usesDefault']== 1) ? true : false;

            if ( $NSArray['hasDefault'] == true ) {
                $data = array();
                $data["id"]           = "ns_usedefaults";
                $data["name"]         = $this->user->lang("Use Defaults");
                $data["isrequired"]   = true;
                $data["ischangeable"] = $this->user->hasPermission('domains_editns');;
                $data["fieldtype"]    = (string)typeYESNO;
                $data["value"]        = @$NSArray['usesDefault'];
                $data["listener"]     = array("onselect"=>"nameservers_ChangeUseDefaults");
                ObjectCustomFields::parseFieldToArray($data, $this->user, $this->settings);
                $datalist[] = $data;
            }

            $cNameServer = 0;
            if (isset($NSArray['nameservers']) && is_array($NSArray['nameservers'])) {
                foreach ($NSArray['nameservers'] as $nameserver) {
                    $cNameServer++;
                    $data = array();
                    $data["id"]           = "ns_".$cNameServer;
                    $data["name"]         = $this->user->lang("Name Server")." ".$cNameServer;
                    $data["isrequired"]   = false;
                    $data["ischangeable"] =  $this->user->hasPermission('domains_editns');;
                    $data["ishidden"]     = $hidefields;
                    $data["width"]        = 172;
                    $data["fieldtype"]    = (string)typeNAMESERVER;
                    $data["value"]        = $nameserver;
                    ObjectCustomFields::parseFieldToArray($data, $this->user, $this->settings);
                    $datalist[] = $data;
                }
            }

            $data = array();
            $data["id"]        = "addnameserver";
            $data["ishidden"]  = $hidefields;
            $data["fieldtype"] = (string)TYPE_UI_BUTTON;
            $data["value"]     = $this->user->lang("Add Name Server");
            $data["listener"]  = array("onclick"=>"nameservers_addnameserver");
            ObjectCustomFields::parseFieldToArray($data, $this->user, $this->settings);
            $datalist[] = $data;

            $this->view->nameservers = $datalist;

        } else {
            $this->view->domainNotActive = true;
        }
    }

    protected function productdomainhostsAction()
    {
        $this->checkPermissions('domains_dnssettings');

        $this->title = $this->user->lang('Domain Host Records');
        $product_id = $this->getParam('id', FILTER_SANITIZE_NUMBER_INT);
        $userPackage = new UserPackage($product_id);

        $this->cssPages = array("templates/default/css/whhg.min.css", "templates/default/css/customfields_public.css", "templates/default/views/clients/productspublic/product.css");
        $this->jsLibs = array('javascript/customfields.js', "templates/default/views/clients/productspublic/productdomainhosts.js", "templates/default/views/clients/productspublic/product.js");

        $this->view->domainNotActive = false;
        $this->view->domainNotRegistered = false;
        $this->view->hostRecordsNotSupported = false;
        $this->view->packageId = $product_id;

        $domainNameGateway = new DomainNameGateway($this->user);
        $isRegistered = $domainNameGateway->isDomainRegistered($userPackage);

        $statusPending = StatusAliasGateway::getInstance($this->user)->getPackageStatusIdsFor(PACKAGE_STATUS_PENDING);
        if ( !$isRegistered ) {
            $this->view->domainNotRegistered = true;
        } else if ($isRegistered && !in_array($userPackage->status, $statusPending)) {

            $this->view->hostRecords = array();

            include_once 'modules/clients/models/ObjectCustomFields.php';
            $dnsArray = array();
            $dnsArray['records'] = array();
            $dnsArray['default'] = 0;

            try {
                $dnsArray = $domainNameGateway->getDNSViaPlugin($userPackage);
            } catch ( Exception $e ) {
                $this->view->hostRecordsNotSupported = true;
            }

            $data = array();
            $data["id"]           = 'blankrecord';
            $data["name"]         = "";
            $data["ischangeable"] = true;
            $data["isrequired"]   = false;
            $data["ishidden"]     = true;
            $data["fieldtype"]    = (string)TYPE_UI_DNSENTRY;
            $data["value"]        = "";
            ObjectCustomFields::parseFieldToArray($data, $this->user, $this->settings);
            $datalist[] = $data;

            foreach ($dnsArray['records'] as $DNS) {
                $data = array();
                $data["id"]              = $DNS['id'];
                $data["name"]            = "";
                $data["isrequired"]      = false;
                $data["ischangeable"]    = ($this->user->isAdmin() ) ? $this->user->hasPermission('clients_edit_customer_packages') : $this->user->hasPermission('domains_updatedns');
                $data["ishidden"]        = false;
                $data["fieldtype"]       = (string)TYPE_UI_DNSENTRY;
                $data["addrowdeletebtn"] = true;
                $data["addrowdeletefnc"] = "hostrecords_deleteaddress";
                $data["value"]           = array("hostname"=>$DNS['hostname'],"hosttype"=>$DNS['type'],"address"=>$DNS['address']);
                ObjectCustomFields::parseFieldToArray($data, $this->user, $this->settings);
                $datalist[] = $data;
            }

            if ($dnsArray['default'] == 1) {
                //show button to register
                //get purchase status
                $data = array();
                $data["id"]           = "addhostrecord";
                $data["name"]         = "";
                $data["isrequired"]   = true;
                $data["ischangeable"] = false;
                $data["ishidden"]     = !$this->user->hasPermission('domains_updatedns');
                $data["fieldtype"]    = (string)TYPE_UI_BUTTON;
                $data["value"]        = $this->user->lang("Add Zone Entry");
                $data["listener"]     = array("onclick"=>"hostrecords_addzoneentry");
                ObjectCustomFields::parseFieldToArray($data, $this->user, $this->settings);
                $datalist[] = $data;
            }
            $this->view->hostRecords = $datalist;

        } else {
            $this->view->domainNotActive = true;
        }
    }

    protected function productbillingAction()
    {
        $this->title = $this->user->lang('Billing');
        $product_id = $this->getParam('id', FILTER_SANITIZE_NUMBER_INT);
        $userPackage = new UserPackage($product_id);

        $this->cssPages = array("templates/default/css/whhg.min.css", "templates/default/css/customfields_public.css", "templates/default/views/clients/productspublic/productbilling.css", "templates/default/views/clients/productspublic/product.css");
        $this->jsLibs = array('javascript/customfields.js', "templates/default/views/clients/productspublic/product.js", "templates/default/views/clients/productspublic/productbilling.js");

        $this->view->productId = $product_id;
        $this->view->displaytoogleautomaticcccharge = false;
        $this->view->checkedtoogleautomaticcccharge = '';
        if ( $this->user->hasPermission('billing_automatic_cc_charge') && $this->user->isAutopayment()) {
            $recurringWork = $userPackage->getRecurringFeeEntry();
            if ($recurringWork->GetRecurring()) {
                $this->view->displaytoogleautomaticcccharge = true;
                $this->view->checkedtoogleautomaticcccharge = $recurringWork->GetAutoChargeCC();
            }
        }

        $userPackageGateway = new UserPackageGateway($this->user);
        $package = new Package($userPackage->Plan);
        $currency = new Currency($this->user);

        $this->view->package = array();

        $languages = CE_Lib::getEnabledLanguages();
        include_once 'modules/admin/models/Translations.php';
        $translations = new Translations();
        $languageKey = ucfirst(strtolower($this->user->getLanguage()));

        if (count($languages) > 1) {
            $this->view->package['desc']         = $translations->getValue(PRODUCT_DESCRIPTION, $package->getId(), $languageKey, $package->description);
        } else {
            $this->view->package['desc']         = $package->description;
        }

        if ($userPackage->getRecurringFeeEntry()->GetRecurring()) {
            $this->view->package['price'] = $currency->format($this->user->getCurrency(), $userPackage->getPrice(false), true, 'NONE', false);

            //Review if there is a Setup fee
            $package->getProductPricing();
            $SetupPrice = $package->getSetupFee($userPackage->getPaymentTerm());
            if ($SetupPrice > 0) {
                $this->view->package['setupprice'] = $currency->format($this->user->getCurrency(), $SetupPrice, true, 'NONE', false);
            } else {
                $this->view->package['setupprice'] = false;
            }
            $this->view->package['setupText'] = $this->user->lang("Setup");
            $this->view->package['termText'] = CE_Lib::getMonthLabel($userPackage->getPaymentTerm(), $this->user);
        } else {
            $this->view->package['price'] = '';

            //Review if there is a One Time option
            $package->getProductPricing();
            $OneTimePrice = $package->getPrice(0);
            if ($OneTimePrice > 0) {
                $this->view->package['price'] = $currency->format($this->user->getCurrency(), $OneTimePrice, true, 'NONE', false);
            }
            $this->view->package['setupprice'] = false;
            $this->view->package['setupText'] = $this->user->lang("Setup");
            $this->view->package['termText'] = CE_Lib::getMonthLabel(0, $this->user);
        }

        $recurringFee = $userPackage->getRecurringFeeEntry();
        $this->view->package['nextBillDate'] = $recurringFee->getNextBillDate();
        if ($this->view->package['nextBillDate'] == "") {
            $this->view->package['nextBillDate'] = $this->user->lang("Not Applicable");
        }

        $addons = $userPackageGateway->getUserPackageAddonFields($userPackage, $languageKey);
        $this->view->addons = $addons;

        $coupons = $userPackageGateway->getUserPackageCouponFields($userPackage);
        $this->view->coupons = $coupons;

        $appliesToPackage = $userPackageGateway->getUserPackageAppliesToPackageFields($userPackage);
        $this->view->appliesToPackage = $appliesToPackage;

    }


    protected function productsslinfoAction()
    {
        $this->title = $this->user->lang('SSL Certificate Information');
        include_once 'modules/admin/models/PluginGateway.php';
        include_once 'modules/clients/models/SSLGateway.php';

        $this->cssPages = array("templates/default/css/whhg.min.css","templates/default/views/clients/productspublic/product.css");
        $this->jsLibs = array("templates/default/views/clients/productspublic/product.js");

        $pluginGateway = new PluginGateway($this->user);
        $sslGateway = new SSLGateway();

        $product_id = $this->getParam('id', FILTER_SANITIZE_NUMBER_INT);
        $userPackage = new UserPackage($product_id);

        $this->view->cert = array();

        // use this to determine if we can change CSR and admin e-mail
        $this->view->cert['isChangeable'] = !$sslGateway->isCertificateIssued($userPackage);
        $certRegistrar = $userPackage->getCustomField("Registrar");

        $sslPlugin = $pluginGateway->getPluginByUserPackage($userPackage, $certRegistrar);
        if ( $this->settings->get('Hide Registrar Information') != 1 || $this->user->isAdmin() ) {
            $this->view->cert['registrar'] = ($certRegistrar == "" || $certRegistrar == "0") ? "None": strtolower($certRegistrar);
        }

        $this->view->cert['type'] = $this->user->lang("Unknown");
        foreach ( $sslGateway->getSSLCertTypes($certRegistrar) as $value ) {
            if ( $value['id'] == $userPackage->getCustomField("Certificate Type") ) {
                $this->view->cert['type'] = $value['name'];
                break;
            }
        }

        $status = $userPackage->getCustomField('Certificate Status');
        $certid = $userPackage->getCustomField("Certificate Id");

        if ( $status != SSL_CERT_ISSUED_STATUS ) {
            if (  $certid != '' ) {
                // Need to catch any Exceptions from here so we still show the Certificate Information
                try {
                    $status = $sslGateway->callPlugin($userPackage, 'doGetCertStatus');
                } catch ( Exception $e ) {
                    $status =  $this->user->lang("Unknown");
                }
            } else {
                $status = $this->user->lang('Unknown');
            }
        }
        $this->view->cert['status'] = $status;

        if ($certid == '') {
            $certid = $this->user->lang("Unknown");
        }
        $this->view->cert['id'] = $certid;

        $domain = $userPackage->getCustomField("Certificate Domain");
        $csr = $userPackage->getCustomField("Certificate CSR");
        if ( $domain == '' ) {
            if ( $csr != '' && $certid != $this->user->lang("Unknown") ) {
                $gateway = new SSLGateway();

                // Need to catch any Exceptions from here so we still show the Certificate Information
                try {
                    $parsedCSR = $gateway->callPlugin($userPackage, 'doParseCSR');
                    $domain = $parsedCSR['domain'];
                    $userPackage->setCustomField("Certificate Domain", $domain);
                } catch ( Exception $e ) {
                    $domain =  $this->user->lang("Unknown");
                }
            } else {
                $domain = $this->user->lang('Unknown');
            }
        }

        $this->view->cert['csr'] = '';
        $this->view->cert['adminEmail'] = '';
        if ( is_object($sslPlugin) && $sslPlugin->usingInviteURL == false ) {
            $this->view->cert['csr'] = $csr;
            $this->view->cert['adminEmail'] = $userPackage->getCustomField("Certificate Admin Email");
        }

        // Do we need to show this?
        //$this->view->serverType = $certWebServerType;
        $this->view->cert['domain'] = $domain;

        $this->view->cert['expirationDate'] = $userPackage->getCustomField("Certificate Expiration Date");
        if ($this->view->cert['expirationDate'] == "") {
            $this->view->cert['expirationDate'] = $this->user->lang('Unknown');
        }

    }

    protected function updateregistrarlockAction()
    {
        $this->checkPermissions('domains_lock');

        $product_id = $this->getParam('id', FILTER_SANITIZE_NUMBER_INT);
        $userPackage = new UserPackage($product_id);

        if ($this->user->getId() != $userPackage->getCustomerId()) {
            CE_Lib::redirectPage("index.php?fuse=clients&controller=products&view=products");
        }

        $value = $this->getParam('value', FILTER_SANITIZE_NUMBER_INT);
        $gateway = new UserPackageGateway($this->user);
        $this->message = $gateway->callPluginAction($userPackage, 'SetRegistrarLock', $value);
        $this->send();
    }

    protected function savedomaincontactinfoAction()
    {
        $this->checkPermissions('domains_updatecontactinfo');

        $product_id = $this->getParam('id', FILTER_SANITIZE_NUMBER_INT);
        $userPackage = new UserPackage($product_id);

        if ($this->user->getId() != $userPackage->getCustomerId()) {
            CE_Lib::redirectPage("index.php?fuse=clients&controller=products&view=products");
        }

        $contactArray = array();
        foreach ($_REQUEST as $key => $value) {
            if ("Registrant_" === substr($key, 0, 11) ) {
                $contactArray[$key] = $value;
            }
        }

        $contactArray['type'] = 'Registrant';
        $dom = DomainNameGateway::splitDomain($userPackage->getCustomField("Domain Name"));
        $contactArray['tld'] = $dom[1];
        $contactArray['sld'] = $dom[0];

        // Make the call
        $gateway = new DomainNameGateway($this->user);
        $gateway->callPlugin($userPackage, 'setContactInformation', $contactArray);

        $this->message = 'Successfully updated contact information.';
        $this->send();
    }

    protected function productdomaincontactinfoAction()
    {
        $this->checkPermissions('domains_viewcontactinfo');
        $this->title = $this->user->lang('Domain Contact Information');
        $product_id = $this->getParam('id', FILTER_SANITIZE_NUMBER_INT);
        $userPackage = new UserPackage($product_id);

        $this->cssPages = array("templates/default/css/whhg.min.css", "templates/default/css/customfields_public.css", "templates/default/views/clients/productspublic/product.css");
        $this->jsLibs = array('javascript/customfields.js', "templates/default/views/clients/productspublic/productdomaincontactinfo.js", "templates/default/views/clients/productspublic/product.js");

        $this->view->domainNotActive = false;
        $this->view->domainNotRegistered = false;
        $this->view->contactInformation = array();
        $this->view->packageId = $product_id;

        $domainNameGateway = new DomainNameGateway($this->user);
        $isRegistered = $domainNameGateway->isDomainRegistered($userPackage);

        $statusPending = StatusAliasGateway::getInstance($this->user)->getPackageStatusIdsFor(PACKAGE_STATUS_PENDING);
        if ( !$isRegistered ) {
            $this->view->domainNotRegistered = true;
        } else if ($isRegistered && !in_array($userPackage->status, $statusPending)) {

            include_once 'modules/clients/models/ObjectCustomFields.php';
            $contactInfoArray = $domainNameGateway->getContactInfoViaPlugin($userPackage);

            foreach ($contactInfoArray as $contactInfo) {

                $data = array();
                $data["id"]           = $contactInfo['field'];
                $data["name"]         = $contactInfo['name'];
                $data["isrequired"]   = false;
                $data["ischangeable"] = "1";
                $data["ishidden"]     = false;
                $data["fieldtype"]    = (string)typeTEXTFIELD;
                $data["value"]        = $contactInfo['value'];
                ObjectCustomFields::parseFieldToArray($data, $this->user, $this->settings);
                $this->view->contactInformation[] = $data;
            }
        } else {
            $this->view->domainNotActive = true;
        }
    }

    public function openpackagedirectlinkAction()
    {
        $this->disableLayout();

        $packageId = $this->getParam('packageId', FILTER_SANITIZE_NUMBER_INT);

        $gateway = new UserPackageGateway($this->user, $this->customer);
        $directLink = $gateway->getPackageDirectLink($packageId);

        if ($directLink !== false) {
            header("Location: ".$directLink['rawlink']);
        }
    }

    public function getproductsAction()
    {

        include_once 'modules/billing/models/Currency.php';

        if (isset($this->session->product_cache)) {
            unset($this->session->product_cache);
        }

        $currency = new Currency($this->user);
        $gateway = new UserPackageGateway($this->user, $this->customer);

        $args = array();
        $args['limit'] = $this->getParam('limit', FILTER_SANITIZE_NUMBER_INT, 5);
        $originalLimit = $args['limit'];
        $args['start'] = $this->getParam('start', FILTER_SANITIZE_NUMBER_INT, 0);
        $originalStart = $args['start'];
        $args['dir'] = $this->getParam('dir', FILTER_SANITIZE_STRING, 'desc');
        $args['sort'] = $this->getParam('sort', FILTER_SANITIZE_STRING, 'id');
        if ($args['sort'] == 'name') {
            $args['start'] = 0;
            $args['limit'] = 0;
        }

        $validSorts = array('id', 'name', 'nextDueDate', 'term', 'status');
        if ( !in_array($args['sort'], $validSorts) ) {
            $args['sort'] = 'id';
        }

        $products = array();

        $activeStatuses = StatusAliasGateway::getInstance($this->user)->getPackageStatusIdsFor(array(PACKAGE_STATUS_ACTIVE));
        $iterator = $gateway->getUserPackagesIterator($this->user->getId(), $args);
        while ($product = $iterator->fetch()) {

            $statusInfo = $gateway->getStyledStatus($product->status);
            $a_product = $product->toArray();
            $languageKey = ucfirst(strtolower($this->user->getLanguage()));
            $a_product['id'] = $product->getId();
            $a_product['name'] = $product->getReference(true, false, '', $languageKey, true);
            $a_product['status'] = $statusInfo['statusText'];
            if ($product->getRecurringFeeEntry()->GetRecurring() && in_array($product->status, $activeStatuses)) {
                $a_product['nextDueDate'] = $this->view->dateRenderer($product->getRecurringFeeEntry()->getNextBillDate());
                $a_product['price'] = $currency->format($this->user->getCurrency(), $product->getPrice(false), true, 'NONE', false);
                $a_product['term'] = CE_Lib::getMonthLabel($product->getPaymentTerm(), $this->user);
            } else {
                $a_product['nextDueDate'] = '----';
                $a_product['price'] = '';
                $a_product['term'] = '----';
            }

            $gateway->getFieldsByProductType($product, $a_product, true);
            $actions = $gateway->getPackageActions($a_product['id']);
            $actionsHtml = '';
            $additionalHTML = '';
            if ( count($actions) > 0 ) {
                $actionsHtml = '<div class="btn-group">
                <a class="btn dropdown-toggle" data-toggle="dropdown" href="#">
                    <i class="icon-cog"></i>
                    <span class="caret"></span>
                </a>
                <ul class="dropdown-menu ce-dropdown-menu dropdown-inverse">';
                foreach ( $actions as $action ) {
                    if ( isset($action['url']) ) {
                        $actionsHtml .= '<li><a href="' . $action['url'] . '">' . $action['name'] . '</a></li>';
                    } else if ( isset($action['form']) ) {
                        $actionsHtml .= $action['link'];
                        $additionalHTML .= $action['form'];
                    } else {
                        $actionsHtml .= "<li><a onclick=\"productview.callPluginAction('" . $action['command'] . "','" . $a_product['id'] . "');\">" . $action['name'] . '</a></li>';
                    }
                }
                $actionsHtml .= '</ul></div>';
                $a_product['actions'] = $actionsHtml . $additionalHTML;
            }
            if ($args['sort'] == 'name') {
                //The package name will be used to sort the array later. The package id is also attached to avoid issues when having different packages with the same name.
                $products[strtolower($a_product['name'].$a_product['id'])] = $a_product;
            } else {
                $products[] = $a_product;
            }
        }

        if ($args['sort'] == 'name') {
            if ($args['dir'] == 'desc') {
                krsort($products);
            } else {
                ksort($products);
            }
            $products = array_slice($products, $originalStart, $originalLimit);
            $products2 = array();
            foreach ($products as $singleProduct) {
                $products2[] = $singleProduct;
            }
            $products = $products2;
        }

        $this->send(array('data' => $products, 'total' => $iterator->getTotalNumItems()));
    }

     /**
      * Public view for products
      *
      * @access protected
      * @return void
      */
    protected function productsAction()
    {
        $this->checkPermissions();

        $this->title = $this->user->lang('My Products');
        $this->cssPages = array("templates/default/views/clients/productspublic/products.css");
        $this->jsLibs = array("templates/default/views/clients/productspublic/product.js", "templates/default/views/clients/productspublic/products.js");
    }

    /**
     * return only the custom fields for a given product
     * @return json
     */
    protected function getproductcustomfieldsAction()
    {

        $packageid = $this->getParam('packageId', FILTER_SANITIZE_NUMBER_INT);

        //we want to get back customfields for a plan we haven't yet saved so we update the field without saving
        $userPackage = new UserPackage($packageid);

        $gateway = new UserPackageGateway($this->user, $this->customer);

        $datalist = $gateway->getUserPackageGeneralFields($userPackage);

        $this->send(
                array(
                    'totalcount' => count($datalist),
                    'productFields' => $datalist
                )
        );
    }

    /**
     * Return the type fields for a product
     *
     * @return json
     */
    protected function getclientproductfieldsAction()
    {

        $packageid = $this->getParam('packageId', FILTER_SANITIZE_NUMBER_INT);
        $settingtype = $this->getParam('settingtype', FILTER_SANITIZE_STRING);

        $userPackage = new UserPackage($packageid);
        // if the user isn't an admin, we need to check to make sure they are looking at their package and not someone elses
        if (!$this->user->isAdmin()) {
            if ($this->user->getId() != $userPackage->CustomerId) {
                throw new CE_ExceptionPermissionDenied("Package does not exist");
            }
        }

        $gateway = new UserPackageGateway($this->user, $this->customer);

        //we should return global errors here that should be visible in all panels
        //Example .. account not created on server (if plugin)
        //Example .. domain not tied to registrar
        //Merge into product fields
        $errorlist = $gateway->getProductError($userPackage);

        //determine what type of fields we want to return
        switch ($settingtype) {
            case 0: //general
                $datalist = array();
                break;
            case 1: //hosting
                //include addons
                $datalist1 = $gateway->getUserPackageHostingFields($userPackage);
                $datalist2 = $gateway->getUserPackageAddonFields($userPackage);
                $datalist = array_merge($datalist1, $datalist2);
                break;
            case 2: //ssl certifications
                $datalist1 = array();
                //$datalist1 = $gateway->getUserPackageCertificateInfoFields($userPackage);
                $datalist2 = $gateway->getUserPackageAddonFields($userPackage);
                $datalist = array_merge($datalist1, $datalist2);
                break;
            case 3: //domain info
                $datalist1 = $gateway->getUserPackageDomainInfoFields($userPackage);
                $datalist2 = $gateway->getUserPackageAddonFields($userPackage);
                $datalist = array_merge($datalist1, $datalist2);
                break;
            case "domaincontactinfo":
                //not sure if we are going to use
                //refactor into plugin itself
                $datalist = $gateway->getUserPackageDomainContactInfoFields($userPackage);
                break;
            case "hostrecords":
                //not sure if we are going to use
                //refactor into plugin itself
                $datalist = $gateway->getUserPackageHostRecordsFields($userPackage);
                break;
            case "nameservers":
                //not sure if we are going to use
                //refactor into plugin itself
                $datalist = $gateway->getUserPackageNameServers($userPackage);
                break;
            case "groupinfo":
                $datalist1 = $gateway->getProductStatus($userPackage, true);
                $datalist2 = $gateway->getProductGroupInformation($userPackage, true);
                $datalist = array_merge($datalist1, $datalist2);
                break;
            case "addons":
                //not sure if we are going to use
                $datalist = $gateway->getUserPackageAddonFields($userPackage);
                break;
            case "billing":
                //not sure if we are going to use
                $datalist = $gateway->getUserPackageBillingFields($userPackage);
                break;
            default:
                throw new Exception("Passing setting type that doesn't exist");
        }

        $products = $gateway->getUserPackages($this->customer->getId(), $packageid);

        $datalist = array_merge($datalist, $errorlist);

        $this->send(
                array(
                    'totalcount' => count($datalist),
                    'productFields' => $datalist,
                    'productinfo' => $products['results'][0]
                )
        );
    }

    public function requestcancellationAction()
    {
        $this->disableLayout();

        $this->checkPermissions('clients_cancel_packages');

        $packageId = $this->getParam('id', FILTER_SANITIZE_NUMBER_INT);
        $type = $this->getParam('type', FILTER_SANITIZE_NUMBER_INT);
        $reason = $this->getParam('reason', FILTER_SANITIZE_STRING);
        $userPackage = new UserPackage($packageId, array(), $this->user);

        if ( $userPackage->CustomerId != $this->user->getId() ) {
            CE_Lib::redirectPermissionDenied($this->user->lang('You can not perform this action on this package'));
        }

        $userPackage->requestCancellation($reason, $type);

        $params = array();
        $params['reason'] = $reason;
        $params['type'] = $type;
        $packageLog = Package_EventLog::newInstance(false, $this->user->getId(), $packageId, PACKAGE_EVENTLOG_CANCELLATIONREQUEST, $this->user->getId(), serialize($params));
        $packageLog->save();

        $customer = new User($userPackage->CustomerId);
        $templategateway = new AutoresponderTemplateGateway();
        $autoresponderTemplate = $templategateway->getEmailTemplateByName("Package Cancellation Requested Template");
        $emailMessage = $autoresponderTemplate->getContents();
        $emailMessageSubject = $autoresponderTemplate->getSubject();
        $autoresponderTemplateID = $autoresponderTemplate->getId();
        if ($autoresponderTemplateID !== false) {
            include_once 'modules/admin/models/Translations.php';
            $languages = CE_Lib::getEnabledLanguages();
            $translations = new Translations();
            $languageKey = ucfirst(strtolower($customer->getRealLanguage()));
            CE_Lib::setI18n($languageKey);
            if (count($languages) > 1) {
                $emailMessageSubject = $translations->getValue(EMAIL_SUBJECT, $autoresponderTemplateID, $languageKey, $emailMessageSubject);
                $emailMessage = $translations->getValue(EMAIL_CONTENT, $autoresponderTemplateID, $languageKey, $emailMessage);
            }
            $emailMessage = str_replace("[PACKAGEREFERENCE]", $userPackage->getReference(true, true, '', $languageKey), $emailMessage);
            $emailMessageSubject = str_replace("[PACKAGEREFERENCE]", $userPackage->getReference(true, true, '', $languageKey), $emailMessageSubject);
            $emailMessage = str_replace("[CLIENTNAME]", $customer->getFullName(true), $emailMessage);
            $emailMessageSubject = str_replace("[CLIENTNAME]", $customer->getFullName(true), $emailMessageSubject);
            $emailMessage = str_replace("[CLIENTAPPLICATIONURL]", CE_Lib::getSoftwareURL(), $emailMessage);
            $emailMessageSubject = str_replace("[CLIENTAPPLICATIONURL]", CE_Lib::getSoftwareURL(), $emailMessageSubject);
            $emailMessage = str_replace("[COMPANYNAME]", $this->settings->get("Company Name"), $emailMessage);
            $emailMessageSubject = str_replace("[COMPANYNAME]", $this->settings->get("Company Name"), $emailMessageSubject);
            $emailMessage = str_replace("[COMPANYADDRESS]", $this->settings->get("Company Address"), $emailMessage);
            $emailMessageSubject = str_replace("[COMPANYADDRESS]", $this->settings->get("Company Address"), $emailMessageSubject);
        }
        include_once 'library/CE/NE_MailGateway.php';
        $MailGateway = new NE_MailGateway();
        $MailGateway->MailMessage(
            $emailMessage,
            $this->settings->get('Support E-mail'),
            $this->settings->get('Company Name'),
            $userPackage->CustomerId,
            '',
            $emailMessageSubject,
            '3',
            '0',
            'notifications',
            '',
            '',
            MAILGATEWAY_CONTENTTYPE_HTML
        );

        CE_Lib::redirectPage('index.php?fuse=clients&controller=products&view=products', $this->user->lang('Your request to cancel has been submitted.'));
    }

    protected function cancelAction()
    {
        $this->title = $this->user->lang('Request Cancellation');
        $this->checkPermissions('clients_cancel_packages');

        $this->jsLibs = array("templates/default/views/clients/productspublic/cancel.js");

        $packageId = $this->getParam('id', FILTER_SANITIZE_NUMBER_INT);
        $userPackage = new UserPackage($packageId);

        if ( $userPackage->CustomerId != $this->user->getId() ) {
            CE_Lib::redirectPermissionDenied($this->user->lang('You can not perform this action on this package'));
        }

        $this->view->packageId = $packageId;
        $this->view->packageName = $userPackage->getReference(true);
        $this->view->endOfBillingPeriod = $userPackage->getEndOfBillingPeriod(false);
        $this->view->paidInvoicesCount = $userPackage->getPaidInvoicesCount();
    }

    public function updatehostingpasswordAction()
    {
        include_once 'modules/clients/models/UserPackageGateway.php';

        $packageId = $this->getParam('id', FILTER_SANITIZE_NUMBER_INT);
        $userPackage = new UserPackage($packageId);

        if ( $userPackage->CustomerId != $this->user->getId() ) {
            CE_Lib::redirectPermissionDenied($this->user->lang('You can not perform this action on this package'));
        }

        $userPackageGateway = new UserPackageGateway($this->user);
        $userPackageGateway->adminUpdateUserPackageHosting($userPackage);

        $params = array();
        $params['Updated Password'] = 'XXX MASKED PASSWORD XXX';
        $packageLog = Package_EventLog::newInstance(false, $this->user->getId(), $packageId, PACKAGE_EVENTLOG_UPDATED, $this->user->getId(), serialize($params));
        $packageLog->save();

        $this->message = $this->user->lang('You have successfully updated your password.');
        $this->send();
    }

    protected function isvalidsubdomainAction()
    {
        $subDomainName = $this->getParam('subDomainName', FILTER_SANITIZE_STRING, "");
        $subDomainTld  = $this->getParam('subDomainTld', FILTER_SANITIZE_STRING, "");
        $gateway = new UserPackageGateway($this->user, $this->customer);
        $result = $gateway->isValidSubDomain($subDomainName, $subDomainTld);
        $this->send(array("results" => $result));
    }

    protected function productsnapinviewAction()
    {
        include_once "modules/admin/models/PluginGateway.php";

        $this->_helper->viewRenderer->setNoRender(true);

        UserGateway::ensureCustomerIsValid($this->customer);
        $this->cssPages = array("templates/default/css/whhg.min.css","templates/default/views/clients/productspublic/product.css");
        $this->jsLibs = array("templates/default/views/clients/productspublic/product.js");

        $snapinName = $this->getParam('name');
        $key = $this->getParam('key');

        $pg = new PluginGateway($this->user);
        $plugin = $pg->getSnapinContent($snapinName, $this->view);

        $view = $pg->getHookByKey($snapinName, 'public_profileproducttab', $key);

        $matched_mapping = array();
        $matched_mapping['type'] = "hooks";
        $matched_mapping['loc'] = "public_profileproducttab";
        $matched_mapping['tpl'] = $view['tpl'];
        $this->title =$view['title'];

        $_GET['selectedtab'] = $snapinName;

        $plugin->setMatching($matched_mapping);

        $loadassets = false;
        $output = $plugin->mapped_view($loadassets);

        echo $this->view->publichelper($this->customer);
        echo $this->view->publicproductheader($this->customer, $output);

        if ($loadassets) {
            //let's see if we have js file and css file for this tab
            if (file_exists("plugins/snapin/".$snapinName."/".$view['tpl'].".js")) {
                $this->jsLibs[] = "plugins/snapin/".$snapinName."/".$view['tpl'].".js";
            }
            if (file_exists("plugins/snapin/".$snapinName."/".$view['tpl'].".css")) {
                $this->cssPages[] = "plugins/snapin/".$snapinName."/".$view['tpl'].".css";
            }
        }
    }
}
